# インポート
すべてのコードを1つのファイルに入れると、すぐに管理が難しくなります。
`import` 文を使うと、他のファイルから関数やグローバル変数をインポートできます。
一目でわかる仕組みはこちら：
![](ImportsInOnePicture400)

ここで `import module2` は `module2` という名前のファイルを実行し、そのすべてのグローバル変数にアクセスできるようにします。
その後、`.` 演算子を使って、インポートしたモジュール内の変数や関数にアクセスできます。
この例では、`module2.print_x()` が `module2` の `print_x()` を呼び出します。

### ここから先は読まなくても大丈夫

`from` 構文を使えば、インポートしたモジュールのグローバル変数を、`import` 文が実行された現在のスコープに移動させることもできます。

`from module2 import print_x
print_x()`
`module2` から指定されたグローバル変数だけをインポートします。

または

`from module2 import *
print_x()`
`module2` からすべてのグローバル変数をインポートします。

これも `module2` ファイルをインポートしますが、`module2` という名前の変数を通してアクセスする代わりに、`module2` のグローバル変数を展開して、ローカルスコープに直接割り当てます。

この形式のインポートは、2つのファイルが互いにインポートし合う場合にうまく機能せず、名前の衝突によってインポート元のファイルで誤って変数を上書きしてしまう可能性があるため、通常は推奨されません。何をしているかよくわからない場合は、`from` 構文を避ける方が安全です。

# 実際の仕組み

## 要約
インポートは直感的でないことがありますが、ほとんどの問題は `from file import` の代わりに `import file` 構文を使い、グローバルな定義以外のすべてを
`if __name__ == "__main__":`
で囲むことで避けられます。

## インポートの副作用
初めてファイルをインポートすると、ファイル全体が実行され、その実行中に定義されたすべての変数にアクセスできるようになります。
同じファイルを再度インポートすると、初回にキャッシュされたモジュールが再度返されるだけです。

これは、`import` 文には副作用がある可能性があることを意味します。`harvest()` を呼び出すファイルをインポートすると、インポート中に実際に収穫が行われます。しかし、再度インポートしても、ファイルは一度しか実行されないため、再び収穫されることはありません。

`__name__` 変数を使えば、このような副作用を避ける方法があります。これは、ファイルが直接実行されたときには自動的に `"__main__"` に設定され、`import` を通して実行されたときにはファイル名に設定される変数です。
ファイルがインポートされたときに実行したくないコードは、`if __name__ == "__main__":` ブロックの中に入れるのが良い習慣とされています。

Pythonでよくあるファイル構造は、ファイル実行時に実行されるべきコードを `main()` 関数の中に入れることです。こうすることで、ローカル変数（`main()` の中で定義）とインポート可能なグローバル変数（`main()` の外で定義）を明確に区別できます。

`a_global_variable = "global"

def main():
    a_local_variable = "local"
    # do things

if __name__ == "__main__":
    main()`

## インポートサイクル
ファイル `a` がファイル `b` をインポートし、ファイル `b` がファイル `a` をインポートするとどうなるでしょうか？

ファイル `a`：
`import b
x = 0`

ファイル `b`：
`import a
def f():
    print(a.x)`

これは問題なく動作します。まだどちらのファイルもロードされておらず、誰かが `import a` を実行したとしましょう。

- `a` は `import b` の行まで実行されます。
- `b` は `import a` の行まで実行されます。
- モジュール `a` はすでに存在しますが、まだ `import b` の行までしか到達していないため、`x` は含まれていません。
- `b` は、ロード途中のモジュール `a` への参照を `a` という名前の変数に保存します。
- `b` は `def` 文を実行し、関数 `f()` を保存します。
- `a` は実行を続け、`x` を初期化します。

誰かが `b.f()` を呼び出すと、`b` が参照しているモジュール `a` が完全にロードされているため、正しく `0` と表示されます。

では、`from` 構文を使った同じコードを考えてみましょう。

ファイル `a`：
`from b import *
x = 0`

ファイル `b`：
`from a import *
def f():
    print(x)`

- `a` は `from b import *` の行まで実行されます。
- `b` は `from a import *` の行まで実行されます。
- モジュール `a` はすでに存在しますが、まだ完全には実行されていません。
- `b` は、現在 `a` にあるすべてのものを自身のグローバルスコープに展開します。この時点で、`a` は `x = 0` の行に到達していないため何も含んでおらず、何もインポートされません。
- `b` は `def` 文を実行し、関数 `f()` を保存します。
- `a` は実行を続け、`x` を初期化します。

もし誰かが `b.f()` を呼び出すと、現在のスコープに `x` が存在しないというエラーが出ます。これは、今回 `b` はまだロード中の `a` への参照を持っておらず、インポート後に追加された定義を見ることができないためです。